<?php
/*--------------------------------------------------------------------------------------------------
    DependencyInjector.php 2019-10-25
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2019 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */

namespace Gambio\StyleEdit;

use ContentIdentificationFactory;
use ContentReadServiceInterface;
use ContentWriteServiceInterface;
use FilesystemAdapter;
use Gambio\StyleEdit\Adapters\ActiveThemeConfigurationAdapter;
use Gambio\StyleEdit\Adapters\Interfaces\CacheCleanerInterface;
use Gambio\StyleEdit\Adapters\Interfaces\LanguageServiceAdapterInterface;
use Gambio\StyleEdit\Adapters\Interfaces\ThemeActivatorAdapterInterface;
use Gambio\StyleEdit\Adapters\Interfaces\ThemeContentImporterAdapterInterface;
use Gambio\StyleEdit\Adapters\LanguageServiceAdapter;
use Gambio\StyleEdit\Adapters\ShopCacheCleaner;
use Gambio\StyleEdit\Adapters\ShopThemeContentImporterAdapter;
use Gambio\StyleEdit\Adapters\ThemeActivatorAdapter;
use Gambio\StyleEdit\Api\Storage\StyleEditExpertModeStorage;
use Gambio\StyleEdit\Configurations\ShopBasePath;
use Gambio\StyleEdit\Configurations\ShopBaseUrl;
use Gambio\StyleEdit\Core\Components\Theme\Entities\Interfaces\ActiveThemeInterface;
use Gambio\StyleEdit\Core\SingletonPrototype;
use GmConfigurationServiceInterface;
use InfoElementContentToThemeJsonConverter;
use InfoElementContentToThemeJsonConverterInterface;
use League\Flysystem\Adapter\Local;
use League\Flysystem\Filesystem;
use MainFactory;
use StaticGXCoreLoader;
use WidgetRegistrar;

/**
 * Class DependencyInjector
 * @package Gambio\StyleEdit\Api
 * @codeCoverageIgnore
 */
class DependencyInjector
{

    /**
     * @throws \Exception
     */
    public static function inject(): void
    {
        /** @var StyleEditConfiguration $settings */
        $settings = SingletonPrototype::instance()->get(StyleEditConfiguration::class);
        self::includeExternalFiles($settings->baseFolderPath());
        
        SingletonPrototype::instance()->setUp(WidgetRegistrar::class,
            static function () {
                return MainFactory::create(WidgetRegistrar::class);
            });
        
        SingletonPrototype::instance()->setUp(\LanguageProviderInterface::class, static function() {
            return MainFactory::create(\LanguageProvider::class, StaticGXCoreLoader::getDatabaseQueryBuilder());
        });
        
        SingletonPrototype::instance()->setUp(ActiveThemeInterface::class,
            static function () {
                return ActiveThemeConfigurationAdapter::create();
            });
        
        SingletonPrototype::instance()->setUp(ThemeContentImporterAdapterInterface::class,
            static function () {
                return new ShopThemeContentImporterAdapter();
            });
        
        SingletonPrototype::instance()->setUp(ContentWriteServiceInterface::class,
            static function () {
                return \StaticGXCoreLoader::getService('ContentWrite');
            });
        SingletonPrototype::instance()->setUp(\ContentIdentificationFactoryInterface::class,
            static function () {
                return MainFactory::create(ContentIdentificationFactory::class);
            });
        
        SingletonPrototype::instance()->setUp(CacheCleanerInterface::class,
            static function () {
                return ShopCacheCleaner::create();
            });
        
        SingletonPrototype::instance()->setUp(StyleEditExpertModeStorage::class, static function() {
            return new StyleEditExpertModeStorage;
        });
        
        SingletonPrototype::instance()->setUp(ContentReadServiceInterface::class,
            static function () {
                return \StaticGXCoreLoader::getService('ContentRead');
            });
        
        SingletonPrototype::instance()->setUp(GmConfigurationServiceInterface::class,
            static function () {
                return \StaticGXCoreLoader::getService('GmConfiguration');
            });
        
        SingletonPrototype::instance()->setUp(LanguageServiceAdapterInterface::class,
            static function () {
                $provider = MainFactory::create(\LanguageProvider::class,
                                                StaticGXCoreLoader::getDatabaseQueryBuilder());
                return new LanguageServiceAdapter($provider);
            });
        
        SingletonPrototype::instance()->setUp(ThemeActivatorAdapterInterface::class,
            static function () {
                return ThemeActivatorAdapter::create();
            });
        SingletonPrototype::instance()->setUp(ShopBaseUrl::class,
            static function () {
                return new ShopBaseUrl((ENABLE_SSL ? HTTPS_SERVER : HTTP_SERVER) . DIR_WS_CATALOG);
            });
        SingletonPrototype::instance()->setUp(ShopBasePath::class,
            static function () {
                return new ShopBasePath(DIR_FS_CATALOG);
            });
        
        SingletonPrototype::instance()->setUp(FilesystemAdapter::class,
            static function () use ($settings) {
                $filesystemAdapter = new Local($settings->themesFolderPath(), LOCK_EX, Local::DISALLOW_LINKS, [
                    'file' => [
                        'public'  => 0777,
                        'private' => 0700,
                    ],
                    'dir'  => [
                        'public'  => 0777,
                        'private' => 0700,
                    ]
                ]);
                $filesystem        = new Filesystem($filesystemAdapter);
                
                return MainFactory::create(FilesystemAdapter::class, $filesystem);
            });
        
        SingletonPrototype::instance()->setUp('FilesystemAdapterShopRoot',
            static function () {
                $filesystemAdapter = new Local(SHOP_ROOT, LOCK_EX, Local::DISALLOW_LINKS, [
                    'file' => [
                        'public'  => 0777,
                        'private' => 0700,
                    ],
                    'dir'  => [
                        'public'  => 0777,
                        'private' => 0700,
                    ]
                ]);
                $filesystem        = new Filesystem($filesystemAdapter);
                
                return MainFactory::create(FilesystemAdapter::class, $filesystem);
            });
        
        SingletonPrototype::instance()->setUp(InfoElementContentToThemeJsonConverterInterface::class, static function() {
            return MainFactory::create(InfoElementContentToThemeJsonConverter::class);
        });
    }
    
    
    private static function includeExternalFiles($basedir)
    {
        require_once $basedir . '/vendor/autoload.php';
        require_once $basedir . '/GXEngine/Shared/Types/StringType.inc.php';
        require_once $basedir . '/GXEngine/Shared/Types/IntType.inc.php';
        require_once $basedir . '/GXEngine/Shared/Types/Interfaces/IdInterface.inc.php';
        require_once $basedir . '/GXEngine/Shared/Types/IdType.inc.php';
        require_once $basedir . '/system/core/MainAutoloader.inc.php';
        require_once $basedir . '/system/core/MainFactory.inc.php';
        require_once $basedir . '/system/core/caching/DataCache.inc.php';
        require_once $basedir . '/GXMainComponents/Services/Core/Content/Services/ContentWriteServiceInterface.inc.php';
    }
    
}